{==============================================================================|
| Project : Ararat Synapse                                       | 001.000.005 |
|==============================================================================|
| Content: SSL support by StreamSecII                                          |
|==============================================================================|
| Copyright (c)1999-2005, Lukas Gebauer                                        |
| All rights reserved.                                                         |
|                                                                              |
| Redistribution and use in source and binary forms, with or without           |
| modification, are permitted provided that the following conditions are met:  |
|                                                                              |
| Redistributions of source code must retain the above copyright notice, this  |
| list of conditions and the following disclaimer.                             |
|                                                                              |
| Redistributions in binary form must reproduce the above copyright notice,    |
| this list of conditions and the following disclaimer in the documentation    |
| and/or other materials provided with the distribution.                       |
|                                                                              |
| Neither the name of Lukas Gebauer nor the names of its contributors may      |
| be used to endorse or promote products derived from this software without    |
| specific prior written permission.                                           |
|                                                                              |
| THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"  |
| AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE    |
| IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE   |
| ARE DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR  |
| ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL       |
| DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR   |
| SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER   |
| CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT           |
| LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY    |
| OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH  |
| DAMAGE.                                                                      |
|==============================================================================|
| The Initial Developer of the Original Code is Lukas Gebauer (Czech Republic).|
| Portions created by Lukas Gebauer are Copyright (c)2005.                     |
| All Rights Reserved.                                                         |
|==============================================================================|
| Contributor(s):                                                              |
|   Henrick Hellstrm <henrick@streamsec.se>                                   |
|==============================================================================|
| History: see HISTORY.HTM from distribution package                           |
|          (Found at URL: http://www.ararat.cz/synapse/)                       |
|==============================================================================}

{:@abstract(SSL plugin for StreamSecII or OpenStreamSecII)

StreamSecII is native pascal library, you not need any external libraries!

You can tune lot of StreamSecII properties by using your GlobalServer. If you not
using your GlobalServer, then this plugin create own TSimpleTLSInternalServer
instance for each TCP connection. Formore information about GlobalServer usage
refer StreamSecII documentation.

If you are not using key and certificate by GlobalServer, then you can use
properties of this plugin instead, but this have limited features and
@link(TCustomSSL.KeyPassword) not working properly yet!

For handling keys and certificates you can use this properties:
@link(TCustomSSL.CertCAFile), @link(TCustomSSL.CertCA),
@link(TCustomSSL.TrustCertificateFile), @link(TCustomSSL.TrustCertificate),
@link(TCustomSSL.PrivateKeyFile), @link(TCustomSSL.PrivateKey),
@link(TCustomSSL.CertificateFile), @link(TCustomSSL.Certificate),
@link(TCustomSSL.PFXFile). For usage of this properties and for possible formats
of keys and certificates refer to StreamSecII documentation.
}

{$IFDEF FPC}
  {$MODE DELPHI}
{$ENDIF}
{$H+}

unit SSL_StreamSec4;

interface

uses
  SysUtils, Classes,
  blcksock, synsock, synautil, synacode,
  StreamSec.Mobile.TlsInternalServer,
  StreamSec.Mobile.TlsConst,
  StreamSec.Mobile.StreamSecII,
  StreamSec.DSI.PkixCert,
  stPkixName,
  stGC,
  stSecUtils,
  StreamSec.Mobile.TlsClass;

type
  {:@abstract(class implementing StreamSecII SSL plugin.)
   Instance of this class will be created for each @link(TTCPBlockSocket).
   You not need to create instance of this class, all is done by Synapse itself!}
  TSSLStreamSec4 = class(TCustomSSL)
  private
    function X500StrToStr(const Prefix, Value: string): string;
    function X501NameToStr(const Value: iName): string;
  protected
    FIsServer: Boolean;
    FTLSServer: TsmCustomTLSInternalServer;
    FServerCreated: Boolean;
    FSocketID: Pointer;
    FClient: TCustomTLS_ContentLayer;
    FInBuffer: TMemoryStream;
    FDataBuffer: TMemoryStream;
    function SSLCheck: Boolean;
    function Init(server:Boolean): Boolean;
    function DeInit: Boolean;
    function Prepare(server:Boolean): Boolean;
    procedure NotTrustEvent(Sender: TObject; const Cert: iCertificate; var ExplicitTrust: Boolean);
    function GetCert: iCertificate;
    procedure RawSend(strm: TCustomMemoryStream);
    procedure RawRecv;
  public
    constructor Create(const Value: TTCPBlockSocket); override;
    destructor Destroy; override;
    {:See @inherited}
    function LibVersion: String; override;
    {:See @inherited}
    function LibName: String; override;
    {:See @inherited and @link(ssl_streamsec) for more details.}
    function Connect: boolean; override;
    {:See @inherited and @link(ssl_streamsec) for more details.}
    function Accept: boolean; override;
    {:See @inherited}
    function Shutdown: boolean; override;
    {:See @inherited}
    function BiShutdown: boolean; override;
    {:See @inherited}
    function SendBuffer(Buffer: TMemory; Len: Integer): Integer; override;
    {:See @inherited}
    function RecvBuffer(Buffer: TMemory; Len: Integer): Integer; override;
    {:See @inherited}
    function WaitingData: Integer; override;
    {:See @inherited}
    function GetSSLVersion: string; override;
    {:See @inherited}
    function GetPeerSubject: string; override;
    {:See @inherited}
    function GetPeerIssuer: string; override;
    {:See @inherited}
    function GetPeerName: string; override;
    {:See @inherited}
    function GetPeerFingerprint: string; override;
    {:See @inherited}
    function GetCertInfo: string; override;
  published
    {:TLS server for tuning of StreamSecII.}
    property TLSServer: TsmCustomTLSInternalServer read FTLSServer write FTLSServer;
  end;

implementation

uses
  stDERCoder,
  stReadStrm;

{==============================================================================}

constructor TSSLStreamSec4.Create(const Value: TTCPBlockSocket);
begin
  inherited Create(Value);
  FClient := nil;
  FIsServer := False;
  FTLSServer := nil;
  FInBuffer := TMemoryStream.Create;
  FDataBuffer := TMemoryStream.Create;
end;

destructor TSSLStreamSec4.Destroy;
begin
  DeInit;
  FInBuffer.Free;
  FDataBuffer.Free;
  inherited Destroy;
end;

function TSSLStreamSec4.LibVersion: String;
begin
  Result := 'StreamSecIV';
end;

function TSSLStreamSec4.LibName: String;
begin
  Result := 'ssl_streamsec4';
end;

function TSSLStreamSec4.SSLCheck: Boolean;
begin
  Result := true;
  FLastErrorDesc := '';
  if not Assigned(FClient) then
    Exit;
  FLastError := FClient.LastAlertCode;
  if FLastError <> 0 then begin
    FLastErrorDesc := StreamSec.Mobile.TlsConst.AlertMsg(FLastError);
  end;
end;

procedure TSSLStreamSec4.NotTrustEvent(Sender: TObject; const Cert: iCertificate; var ExplicitTrust: Boolean);
begin
  ExplicitTrust := true;
end;

function TSSLStreamSec4.Init(server:Boolean): Boolean;
begin
  Result := False;
  FIsServer := Server;
  if Assigned(FTLSServer) then begin
    Result := True
  end else begin
    if Assigned(StreamSec.Mobile.TLSInternalServer.GlobalServer) then begin
      FTLSServer := StreamSec.Mobile.TLSInternalServer.GlobalServer;
      Result := True;
    end;
  end;
  if Result then begin
    if server then
      Result := FTLSServer.ClientOrServer = cosServerSide
    else
      Result := FTLSServer.ClientOrServer = cosClientSide;
  end;
  if Result then begin
    if server then
      FClient := FTLSServer.TLSAddServerSession
    else
      FClient := FTLSServer.TLSAddClientSession;
  end;
end;

function TSSLStreamSec4.DeInit: Boolean;
var
  strm: TMemoryStream;
begin
  Result := True;
  if Assigned(FClient) then begin
    if FLastError = 0 then begin
      strm := TMemoryStream.Create;
      try
        FClient.Close(strm);
        RawSend(strm);
      finally
        strm.Free;
      end;
    end;
    FClient.Release;
    FClient := nil;
  end;
  FSSLEnabled := false;
end;

function TSSLStreamSec4.Prepare(server:Boolean): Boolean;
begin
  Result := false;
  DeInit;
  if Init(server) then
    Result := true
  else
    DeInit;
end;

function TSSLStreamSec4.Connect: boolean;
var
  outstrm: TMemoryStream;
begin
  Result := False;
  if FSocket.Socket = INVALID_SOCKET then
    Exit;
  if Prepare(false) then begin
    FDataBuffer.SetSize(0);
    outstrm := TMemoryStream.Create;
    try
    if vsnDNS in FTLSServer.Options.VerifyServerName then
        FClient.DNSNameToCheck := SNIHost;
      FClient.Connect(outstrm);
      RawSend(outstrm);
      while not FClient.Encrypted do begin
        SSLCheck;
        if FLastError <> 0 then
          Exit;
        RawRecv;
        if FLastError <> 0 then
          Exit;
        outstrm.Position := 0;
        FClient.DecodeData(FInBuffer,FDataBuffer,outstrm);
        if outstrm.Position > 0 then
          RawSend(outstrm);
      end;
    finally
      outstrm.Free;
    end;
    FSSLEnabled := FClient.Encrypted;
    Result := FSSLEnabled;
    FDataBuffer.Position := 0;
  end;
end;

function TSSLStreamSec4.Accept: boolean;
var
  outstrm: TMemoryStream;
begin
  Result := False;
  if FSocket.Socket = INVALID_SOCKET then
    Exit;
  if Prepare(true) then begin     
    FDataBuffer.SetSize(0);
    outstrm := TMemoryStream.Create;
    try
      RawRecv;  
      if FLastError <> 0 then
        Exit;
      FClient.Accept(FInBuffer,outstrm);
      RawSend(outstrm);
      while not FClient.Encrypted do begin
        SSLCheck;
        if FLastError <> 0 then
          Exit;
        RawRecv;
        if FLastError <> 0 then
          Exit;
        outstrm.Position := 0;
        FClient.DecodeData(FInBuffer,FDataBuffer,outstrm);
        if outstrm.Position > 0 then
          RawSend(outstrm);
      end;
    finally
      outstrm.Free;
    end;
    FSSLEnabled := FClient.Encrypted;
    Result := FSSLEnabled;
    FDataBuffer.Position := 0;
  end;
end;

function TSSLStreamSec4.Shutdown: boolean;
begin
  Result := BiShutdown;
end;

function TSSLStreamSec4.BiShutdown: boolean;
begin
  DeInit;
  Result := True;
end;

function TSSLStreamSec4.SendBuffer(Buffer: TMemory; Len: Integer): Integer;
var
  instrm, outstrm: TCustomMemoryStream;
begin
  Result := 0;
  instrm := TReadStream.Create(Buffer^,Len);
  try
    outstrm := TMemoryStream.Create;
    try
      while instrm.Position < Len do begin
        outstrm.Size := 0;
        FClient.EncodeData(instrm,outstrm);
        SSLCheck;
        if FLastError <> 0 then
          Exit;
        RawSend(outstrm);
        if FLastError <> 0 then
          Exit;
      end;
    finally
      outstrm.Free;
    end;
  finally
    instrm.Free;
  end;
  Result := Len;
end;

procedure TSSLStreamSec4.RawRecv;
var
  x, lLen: Integer;
  lResult: Integer;
begin
  if FInBuffer.Position = FInBuffer.Size then
    if synsock.IoctlSocket(FSocket.Socket, FIONREAD, x) = 0 then begin
      FInBuffer.Position := 0;
      lLen := $8000;
      if FInBuffer.Size < lLen then
        FInBuffer.SetSize(lLen);
      lResult := Recv(FSocket.Socket,FInBuffer.Memory,lLen,0);
      if lResult = SOCKET_ERROR then begin
        FLastErrorDesc := '';
        FLastError := WSAGetLastError;
      end else
        FInBuffer.SetSize(lResult);
    end;
end;

procedure TSSLStreamSec4.RawSend(strm: TCustomMemoryStream);
var
  lResult: Integer;
begin
  lResult := Send(FSocket.Socket,strm.Memory,strm.Position,0);
  if lResult=SOCKET_ERROR then begin
    FLastErrorDesc:='';
    FLastError:=WSAGetLastError;
  end;
end;

function TSSLStreamSec4.RecvBuffer(Buffer: TMemory; Len: Integer): Integer;
begin
  Result := FDataBuffer.Read(Buffer^,Len);
end;

function TSSLStreamSec4.WaitingData: Integer;
var
  outstrm: TMemoryStream;
begin
  Result := 0;
  while (Result = 0) and (FDataBuffer.Position = FDataBuffer.Size) do begin
    RawRecv;
    if FLastError <> 0 then
      Exit;
    if FInBuffer.Position < FInBuffer.Size then begin
      outstrm := TMemoryStream.Create;
      try
        FDataBuffer.Position := 0;
        FClient.DecodeData(FInBuffer,FDataBuffer,outstrm);
        FDataBuffer.SetSize(FDataBuffer.Position);
        FDataBuffer.Position := 0;
        if outstrm.Size > 0 then
          RawSend(outstrm);
      finally
        outstrm.Free;
      end;
      Result := FDataBuffer.Size - FDataBuffer.Position;
    end else
      Break;
  end;
  Result := FDataBuffer.Size - FDataBuffer.Position;
end;

function TSSLStreamSec4.GetSSLVersion: string;
begin
  Result := 'SSLv3 or TLSv1';
end;

function TSSLStreamSec4.GetCert: iCertificate;
begin
  if FIsServer then
    Result := FClient.ClientCertificate
  else
    Result := FClient.ServerCertificate;
end;

function TSSLStreamSec4.GetPeerSubject: string;
var
  lCert: iCertificate;
begin
  Result := '';
  lCert := GetCert;
  if Assigned(lCert) then begin
    Result := X501NameToStr(lCert.tbsCertificate.subject);
  end;
end;

function TSSLStreamSec4.GetPeerName: string;
var
  lCert: iCertificate;
begin
  Result := '';
  lCert := GetCert;
  if Assigned(lCert) then begin
    Result := lCert.tbsCertificate.subject.AsRdnSequence.commonName;
  end;
end;

function TSSLStreamSec4.GetPeerIssuer: string;
var
  lCert: iCertificate;
begin
  Result := '';
  lCert := GetCert;
  if Assigned(lCert) then begin
    Result := X501NameToStr(lCert.tbsCertificate.issuer);
  end;
end;

function TSSLStreamSec4.GetPeerFingerprint: string;
var
  lCert: iCertificate;
begin
  Result := '';
  lCert := GetCert;
  if Assigned(lCert) then
    Result := MD5(lCert.GetStruct.EncodedContentsOS(nil,True));
end;

function TSSLStreamSec4.GetCertInfo: string;
var
  Cert: iCertificate;
  l: Tstringlist;
begin
  Result := '';
  Cert := GetCert;
  if Assigned(cert) then
  begin
    l := TStringList.Create;
    try
      Cert.GetStruct.ContentAsStrings(l,1);
      Result := l.Text;
    finally
      l.free;
    end;
  end;
end;

function TSSLStreamSec4.X500StrToStr(const Prefix: string;
  const Value: string): string;
begin
  if Value = '' then
    Result := ''
  else
    Result := '/' + Prefix + '=' + Value;
end;

function TSSLStreamSec4.X501NameToStr(const Value: iName): string;
begin
  Result := X500StrToStr('CN',Value.AsRdnSequence.commonName) +
           X500StrToStr('C',Value.AsRdnSequence.countryName) +
           X500StrToStr('L',Value.AsRdnSequence.localityName) +
           X500StrToStr('ST',Value.AsRdnSequence.stateOrProvinceName) +
           X500StrToStr('O',Value.AsRdnSequence.organizationName) +
           X500StrToStr('OU',Value.AsRdnSequence.organizationalUnitName) +
           X500StrToStr('T',Value.AsRdnSequence.title) +
           X500StrToStr('N',Value.AsRdnSequence.name) +
           X500StrToStr('G',Value.AsRdnSequence.givenName) +
           X500StrToStr('I',Value.AsRdnSequence.initials) +
           X500StrToStr('SN',Value.AsRdnSequence.surname) +
           X500StrToStr('GQ',Value.AsRdnSequence.generationQualifier) +
           X500StrToStr('DNQ',Value.AsRdnSequence.dnQualifier) +
           X500StrToStr('E',Value.AsRdnSequence.emailAddress);
end;


{==============================================================================}

initialization
  SSLImplementation := TSSLStreamSec4;

finalization

end.

